Attribute VB_Name = "gearDesign"
'	This is a part of the source code for Pro/DESKTOP.
'	Copyright (C) 1999 Parametric Technology Corporation.
'	All rights reserved.

'A Visual Basic Example to create a parameterized Gear Design.
'The example also selects the necessary edges(root of the teeth) and carries out subsequent blend operations.

'Global data
Dim module As Double
Dim Dp As Double            'Dp = Dia of pinion
Dim Dg As Double            'Dg = Dia of gear
Dim Tp As Integer           'Tp = No of teeth of pinion
Dim Tg As Integer           'Tg = No of teeth of gear
Dim thickness As Double     'thickness of gear and pinion
Dim shaft1dia As Double
Dim shaft2dia As Double
Dim shaftheight As Double

Dim app As ProDESKTOP
Dim gear1 As PartDocument
Dim gear2 As PartDocument
Dim shaft1 As PartDocument
Dim shaft2 As PartDocument
Dim gearAssembly1 As PartDocument

'Input parameters
Dim p As Double
Dim Np As Double
Dim vr As Double
Dim Sut As Double
Dim Bs As Double

'Global constants
Const Pi As Double = 3.14159265359

'Define VectorClass, MatrixClass and DirectionClass
Dim vecCls As VectorClass
Dim matrixCls As MatrixClass
Dim dirCls As DirectionClass

Rem Master Subroutine that calls all other routines.
Public Sub mainGearDesignFunction()

        Rem To create the global Application Object
        GetApplicationObject

        'Create VectorClass, MatrixClass and DirectionClass
        Set vecCls = app.GetClass("Vector")
        Set dirCls = app.GetClass("Direction")
        Set matrixCls = app.GetClass("Matrix")
        
        Rem To get the input gear parameters and the file path to store the parts
        getInputData

        Rem Routine that designs the gears and gets the necessary parameters for creating a model
        geardesign

        Rem Routine that designs the shafts and gets the necessary parameters for creating a model
        Shaftdesign

        Rem modelGear creates the gear in ProDESKTOP
        Set gear1 = ModelGear(module, Dp, Tp, thickness, shaft1dia)
        BlendGear gear1

        Rem modelShaft creates the shaft in ProDESKTOP
        Set shaft1 = modelShaft(shaft1dia)

        Set gear2 = ModelGear(module, Dg, Tg, thickness, shaft2dia)
        BlendGear gear2

        Set shaft2 = modelShaft(shaft2dia)

        MsgBox "Gear Design Over"

End Sub

Private Sub GetApplicationObject()
        If app Is Nothing Then
                Set app = CreateObject("ProDESKTOP.Application")
                app.SetVisible True
        End If

End Sub

Private Sub getInputData()

        'Power Transmitted in Kilowatts
        p = 10

        'Pinion Speed in rpm
        Np = 1440

        'No of teeth in Pinion
        Tp = 14

        'velocity ratio
        vr = 2

        'Allowable Ultimate stresses for Pinion and Gear materials in N/mm2
        Sut = 600

        'Bending Stress Bs
        Bs = 200

End Sub

Private Sub geardesign()

        'Pi = 3.14159265359

        'Tg is the no of teeth in gear
        Tg = Tp * vr

        'Formula to calculate the Bending Moment
        Mt = (60 * (10 ^ 6) * p) / (2 * Pi * Np)

        'Module list
        Dim m
        m = Array(1#, 1.25, 1.5, 2#, 2.5, 3#, 4#, 5#, 6#, 8#, 10, 12, 16, 20)

        'Design of the pinion
        'To find the module of Pinion

        'tooth form factor yp for pinion and yg for gear
        yp = 0.308

        'Service factor ( Ratio of starting torque to actual torque
        Cs = 1.5

        Fs = 1.5

        'Trial pitch line velocity for pinion
        v = 5
        'Velocity factor Cv
        Cv = 3 / (3 + v)

        'Breadth to module ratio
        btom = 10

        temp = (60 * (10 ^ 6)) / Pi

        'To find the module
        module = (temp * ((p * Cs * Fs) / (Tp * Np * Cv * btom * (Sut / 3) * yp))) ^ (1 / 3)

        'Standardise Module
        For i = 1 To 14
                If module > m(i) And module < m(i + 1) Then
                        module = m(i + 1)
                        J = i + 1
                        Exit For
                End If
        Next i

        For i = 1 To 14
                Dp = module * Tp
                Dg = module * Tg
                thickness = module * btom
                'Check for design
                pt = 2 * Mt / Dp
                v = Pi * Dp * Np / (60 * 10 ^ 3)
                Cv = 3 / (3 + v)
                Peff = Cs * pt / Cv
                Sb = module * thickness * Bs * yp
    
                'factor of Safety Fs
                Fsnew = Sb / Peff
    
                If Fsnew > Fs Then
                        Exit For
                Else
                        module = m(J + 1)
                End If
        Next i

        'Converting mm to m
        Dp = Dp / 1000
        Dg = Dg / 1000
        module = module / 1000
        thickness = thickness / 1000

End Sub

Private Sub Shaftdesign()

        'A simple design of shaft is adopted. this may be replaced by more complicated design routines
        shaft1dia = Dp / 3
        shaft2dia = Dg / 3
        shaftheight = 0.2

End Sub

' Function to Create the Grear.
Private Function ModelGear(module As Double, Pitchdia As Double, Teeth As Integer, thick As Double, shaftdia As Double)

        GetApplicationObject

        Dim part As PartDocument
        Set part = app.NewPart

        MsgBox "Creating the Gear"

        Dim design As aDesign
        Set design = part.GetDesign

        pcd = Pitchdia
        m = module

        Dim T As Double
        T = Teeth
        ocd = pcd + 2 * m
        icd = pcd - 2.5 * m
        pcr = pcd / 2
        ocr = ocd / 2
        icr = icd / 2

        shaftrad = shaftdia / 2

        'Pi = 3.14159265359

        'degtheta is the pressure angle
        degtheta = 20
        radtheta = Pi * 20 / 180

        'x is the offset between the top edge of tooth and the line thru the
        'end point of the tooth at the center
        x = m * Tan(radtheta)

        'twidth is the tooth width
        twidth = 1.5708 * m

        halfwidth = twidth / 2

        'base is the basewidth
        Base = twidth + 2 * x

        'Top is the topwidth
        Dim top As Double
        top = twidth - 2 * x

        'theta is angle subtended by one tooth + one cavity at the center
        Dim theta As Double
        theta = ((2 * Pi) / T)

        'theta1 is the angle subtended by the base edge of the teeth
        theta1 = ((2 * Pi) / T) * (Base / (Base + top))

        'theta2 is the angle subtended by the top edge of the teeth
        theta2 = ((2 * Pi) / T) * (top / (Base + top))

        Dim pointorigin As zVector
        Set pointorigin = vecCls.CreateVector(0, 0, 0)

        'Create the Sketch on which the Gear will be created.
        Dim wp As aWorkplane
        Set wp = part.GetActiveWorkplane

        Dim plane1 As zPlane
        Set plane1 = wp.GetGeometry

        Dim sketch1 As aSketch
        Set sketch1 = part.GetActiveSketch

        part.SetActiveSketch sketch1

        Dim api As helm
        Set api = app.TakeHelm

        'draw the circles
        'Dedendum circle
        Dim zinnercircle As zCurve
        Set zinnercircle = app.GetClass("BasicCircle").CreateBasicCircle(pointorigin, plane1.GetNormal, icr)
        Dim ainnercircle As aLine
        Set ainnercircle = sketch1.CreateLine(zinnercircle)

        api.CommitCalls "Create Circle1", pause

        'To draw the profile of a single tooth of the gear in another sketch
        Dim sketch2 As aSketch
        Set sketch2 = wp.CreateSketch("sketch2")

        part.SetActiveSketch sketch2

        Dim point1 As zVector
        Dim point2 As zVector
        Dim pointA As zVector
        Dim pointB As zVector
        Dim pointC As zVector
        Dim pointD As zVector
        Dim helppoint1 As zVector
        Dim helppoint2 As zVector

        'Initialisations
        xorigin = 0#
        yorigin = 0#
        zorigin = 0#

        'Set the basic points

        Set point1 = vecCls.CreateVector(xorigin, yorigin + ocr, zorigin)
        Set point2 = vecCls.CreateVector(xorigin, yorigin + icr, zorigin)

        'Set the derived points

        Set pointA = point1.rotate(plane1.GetNormal, -theta2 / 2)
        Set pointB = point1.rotate(plane1.GetNormal, theta2 / 2)
        Set helppoint1 = point2.rotate(plane1.GetNormal, -theta1 / 2)
        Set helppoint2 = point2.rotate(plane1.GetNormal, theta1 / 2)
        Set pointC = zinnercircle.ProjectPoint(helppoint1, True)
        Set pointD = zinnercircle.ProjectPoint(helppoint2, True)

        'Create a BasicStraight
        Dim basicStrCls As BasicStraightClass
        Set basicStrCls = app.GetClass("BasicStraight")
        
        Dim curve1 As ZStraight
        Set curve1 = basicStrCls.CreateBasicStraightTwoPoints(pointC, pointA)

        Dim line1 As aLine
        Set line1 = sketch2.CreateLine(curve1)

        Dim curve2 As ZStraight
        Set curve2 = basicStrCls.CreateBasicStraightTwoPoints(pointA, pointB)

        Dim line2 As aLine
        Set line2 = sketch2.CreateLine(curve2)

        Dim curve3 As ZStraight
        Set curve3 = basicStrCls.CreateBasicStraightTwoPoints(pointB, pointD)

        Dim line3 As aLine
        Set line3 = sketch2.CreateLine(curve3)

        Dim curve4 As zBasicStraight
        Set curve4 = basicStrCls.CreateBasicStraightTwoPoints(pointD, pointC)

        Dim line4 As aLine
        Set line4 = sketch2.CreateLine(curve4)

        'Create a SetClass
        Dim setCls As ObjectSetClass
        Set setCls = app.GetClass("ObjectSet")

        Dim obset1 As ObjectSet
        Set obset1 = setCls.CreateAObjectSet

        obset1.AddMember line1
        obset1.AddMember line2
        obset1.AddMember line3
        obset1.AddMember line4

        wp.AutoConstrain obset1

        api.CommitCalls "Create Tooth1", pause

        'Make circular duplicate of the single tooth profile to get the whole gear profile
        CircularDuplicate T, 360, True, obset1
        api.CommitCalls "Create GearProfile", pause

        'Create a ExtrusionClass
        Dim extrusionCls As ExtrusionClass
        Set extrusionCls = app.GetClass("Extrusion")

        'To create the cylindrical extrusion
        Dim extrusion1 As aExtrusion
        Set extrusion1 = extrusionCls.CreateExtrusion(part.GetDesign, sketch1, thick, 0, 0, 0, 1, 0)
        part.UpdateDesign
        'api.CommitCalls "Create Extrusion1", pause

        'To create extrusion for the gear profile
        Dim extrusion2 As aExtrusion
        Set extrusion2 = extrusionCls.CreateExtrusion(part.GetDesign, sketch2, thick, 0, 0, 0, 1, 0)
        part.UpdateDesign
        api.CommitCalls "Create Extrusion2", pause

        'Shaft hole
        Dim sketch3 As aSketch
        Set sketch3 = wp.CreateSketch("sketch3")

        'Create a BasicCircleClass
        Dim basicCirCls As BasicCircleClass
        Set basicCirCls = app.GetClass("BasicCircle")

        Dim zShaftCircle As zCurve
        Set zShaftCircle = basicCirCls.CreateBasicCircle(pointorigin, plane1.GetNormal, shaftrad)

        Dim aShaftCircle As aLine
        Set aShaftCircle = sketch3.CreateLine(zShaftCircle)

        'To create the shaft hole
        Dim extrusion3 As aExtrusion
        Set extrusion3 = extrusionCls.CreateExtrusion(part.GetDesign, sketch3, thick, 0, 0, 0, 2, 0)
        part.UpdateDesign
        api.CommitCalls "Create Extrusion", pause

        MsgBox "Gear Created"

        Set ModelGear = part

End Function

Private Sub BlendGear(part As PartDocument)

        GetApplicationObject

        MsgBox "Blending the gears"
        Set design = part.GetDesign
        Dim wp1 As aWorkplane
        Set wp1 = part.GetActiveWorkplane

        Dim api As helm
        Set api = app.TakeHelm

        Dim plane1 As zPlane
        Set plane1 = wp1.GetGeometricForm
        Dim norm As zDirection
        Set norm = plane1.GetNormal

        'Create an OffsetPlaneClass
        Dim offsetPlaneCls As OffsetPlaneClass
        Set offsetPlaneCls = app.GetClass("OffsetPlane")

        'Create an offset zPlane
        Dim OffPlane As zPlane
        Set OffPlane = offsetPlaneCls.CreateOffsetPlane(plane1, thickness)
    
        'Create the Offset Workplane
        Dim wp2 As aWorkplane
        Set wp2 = design.CreateWorkplane(OffPlane, "topFacePlane")

        'To select the edges of gear which are perpendicular to the base workplane for blending
        MsgBox "Automatic Selection of edges for blending"
        Dim edgeset As ObjectSet
        Set edgeset = design.GetEdges
        count = edgeset.GetCount
        Dim edgeset1 As ObjectSet
        Set edgeset1 = design.GetEdgesInPlane(wp1.GetGeometry, False)
        Dim edgeset2 As ObjectSet
        Set edgeset2 = design.GetEdgesInPlane(wp2.GetGeometry, False)

        Call RemoveSet(edgeset, edgeset1)
        Call RemoveSet(edgeset, edgeset2)

        'edgeset.RemoveSet edgeset1
        'edgeset.RemoveSet edgeset2

        part.SetSelection edgeset

        'MsgBox "Blending the Selected edges"

        Dim rad As Double
        rad = 0.002

        'Create an BlendClass
        Dim blendCls As BlendClass
        Set blendCls = app.GetClass("Blend")

        Dim blend1 As aBlend
        Set blend1 = blendCls.CreateBlend(design, edgeset, 0, 0, rad, 0)

        part.UpdateDesign

        api.CommitCalls "Blend", pause

        MsgBox "Blending done"

End Sub

Private Function modelShaft(shaftdia As Double)

        GetApplicationObject

        Dim part As PartDocument
        Set part = app.NewPart

        MsgBox "Creating the Shaft"
        shaftrad = shaftdia / 2

        Dim wp As aWorkplane
        Set wp = part.GetActiveWorkplane

        Dim plane1 As zPlane
        Set plane1 = wp.GetGeometricForm

        Dim sketch1 As aSketch
        Set sketch1 = part.GetActiveSketch

        Dim api As helm
        Set api = app.TakeHelm

        Dim origin As zVector
        Set origin = vecCls.CreateVector(0, 0, 0)

        'Create a BasicCircleClass
        Dim basicCirCls As BasicCircleClass
        Set basicCirCls = app.GetClass("BasicCircle")

        Dim zcircle1 As zCircle
        Set zcircle1 = basicCirCls.CreateBasicCircle(origin, plane1.GetNormal, shaftrad)

        Dim acircle1 As aLine
        Set acircle1 = sketch1.CreateLine(zcircle1)
        api.CommitCalls "Create Circle", pause

        'Create an ExtrusionClass
        Dim extrusionCls As ExtrusionClass
        Set extrusionCls = app.GetClass("Extrusion")

        Dim extrusion1 As aExtrusion
        Set extrusion1 = extrusionCls.CreateExtrusion(part.GetDesign, sketch1, shaftheight, 0, 0, 0, 1, 0)
        part.UpdateDesign
        api.CommitCalls "Create Extrusion", pause

        MsgBox "Shaft Created"
        Set modelShaft = part

End Function

'=========================================================================================================
'Convenience Functions used by the GearDesign Application

'Function to create a circular duplicate
Private Sub CircularDuplicate(number As Double, dpangle As Double, IsTotalpangle As Boolean, objset As ObjectSet)

        'To get the global objects like application, part, workplane, sketch
        GetApplicationObject
        Dim part As PartDocument
        Set part = app.GetActiveDoc
        Dim Sketch As aSketch
        Set Sketch = part.GetActiveSketch
        Dim wp As aWorkplane
        Set wp = part.GetActiveWorkplane

        Dim pangle As Double

        If (IsTotalpangle = True) Then
                pangle = dpangle / number
        Else
                pangle = dpangle
        End If

        pangle = pangle * Pi / 180

        Dim localZv As zVector
        Set localZv = wp.GetLocalX.Cross(wp.GetLocalY)
        Dim localZ As zDirection
        Set localZ = dirCls.CreateDirection(localZv.GetAt(0), localZv.GetAt(1), localZv.GetAt(2))

        Dim x As zVector
        Dim y As zVector

        Set x = vecCls.CreateVector(1, 0, 0)
        Set y = vecCls.CreateVector(0, 1, 0)

        Dim localOrigin As zVector
        Set localOrigin = wp.GetLocalOrigin

        Dim removeorigin As zMatrix
        Set removeorigin = matrixCls.CreateTranslationMatrix(localOrigin.GetNegative)

        Dim addorigin As zMatrix
        Set addorigin = matrixCls.CreateTranslationMatrix(localOrigin)

        For i = 1 To number - 1

                Dim rotate As zMatrix
                Dim directionCls As DirectionClass
                Set directionCls = app.GetClass("Direction")
                Dim dir1 As zDirection
                Dim dir2 As zDirection
                Dim vect1 As zVector
                Set vect1 = x.rotate(localZ, pangle * i)
                Set dir1 = directionCls.CreateDirection(vect1.GetAt(0), vect1.GetAt(1), vect1.GetAt(2))
                Set vect1 = y.rotate(localZ, pangle * i)
                Set dir2 = directionCls.CreateDirection(vect1.GetAt(0), vect1.GetAt(1), vect1.GetAt(2))
                Set rotate = matrixCls.CreateRotationMatrix(dir1, dir2)
                Dim trans As zMatrix
                Set trans = addorigin.MultiplyByMatrix(rotate).MultiplyByMatrix(removeorigin)

                duplicatelines Sketch, objset, trans

        Next i

End Sub

Private Sub duplicatelines(sk As aSketch, objset As ObjectSet, trans)

        'Create an ItClass
        Dim itCls As ItClass
        Set itCls = app.GetClass("It")
        
        Dim it As Iterator
        Set it = app.GetClass("It").CreateAObjectIt(objset)

        Dim obj
        Set obj = it.start()

        Do While it.IsActive

                Dim curve As zCurve
                Set curve = obj.GetGeometry.Clone
    
                curve.transform trans
    
                Dim line As aLine
                Set line = sk.CreateLine(curve)
    
                Set obj = it.Next()
    
        Loop

End Sub

Private Sub RemoveSet(destinationSet, sourceSet)

        'Create an ItClass
        Dim itCls As ItClass
        Set itCls = app.GetClass("It")

        Dim it As Iterator
        Set it = itCls.CreateAObjectIt(sourceSet)
        destinationSet.RemoveMember it.start
        Do While it.IsActive
                destinationSet.RemoveMember it.Current
                it.Next
        Loop

End Sub


